
import { NextRequest, NextResponse } from 'next/server'
import axios from 'axios'
import * as cheerio from 'cheerio'
import probe from 'probe-image-size'
import slugify from 'slugify'
import OpenAI from 'openai'

const OPENAI = new OpenAI({ apiKey: process.env.OPENAI_API_KEY })

const MIN_W = Number(process.env.MIN_IMG_WIDTH || 600)
const MIN_H = Number(process.env.MIN_IMG_HEIGHT || 600)

const BING_SEARCH_KEY = process.env.BING_SEARCH_KEY as string
const BING_SEARCH_ENDPOINT = process.env.BING_SEARCH_ENDPOINT || 'https://api.bing.microsoft.com/v7.0/search'
const BING_IMAGE_ENDPOINT = process.env.BING_IMAGE_ENDPOINT || 'https://api.bing.microsoft.com/v7.0/images/search'

function psSlug(s: string) {
  return slugify(s, { lower: true, strict: true, locale: 'es' }).slice(0, 128)
}

async function webSearch(q: string) {
  const { data } = await axios.get(BING_SEARCH_ENDPOINT, {
    headers: { 'Ocp-Apim-Subscription-Key': BING_SEARCH_KEY },
    params: { q, count: 10, mkt: 'es-ES', setLang: 'es' }
  })
  return (data.webPages?.value || []).map((r: any) => ({ url: r.url, name: r.name }))
}

async function imageSearch(q: string) {
  const { data } = await axios.get(BING_IMAGE_ENDPOINT, {
    headers: { 'Ocp-Apim-Subscription-Key': BING_SEARCH_KEY },
    params: { q, count: 15, imageType: 'Photo', color: 'ColorOnly', safeSearch: 'Moderate' }
  })
  return (data.value || []).map((i: any) => ({ url: i.contentUrl, width: i.width, height: i.height }))
}

async function validateWhiteBgAndSize(url: string) {
  try {
    const stream = await axios.get(url, { responseType: 'stream', timeout: 10000 })
    const info = await probe(stream.data)
    stream.data.destroy?.()
    if (!info?.width || !info?.height) return false
    if (info.width < MIN_W || info.height < MIN_H) return false
    return true
  } catch { return false }
}

async function extractText(url: string) {
  try {
    const { data } = await axios.get(url, { timeout: 15000 })
    const $ = cheerio.load(data)
    $('script,style,noscript').remove()
    const text = $('body').text().replace(/\s+/g, ' ').trim()
    const bullets = $('li').map((_, el) => $(el).text().trim()).get().filter(Boolean).slice(0, 200)
    return { text: text.slice(0, 50000), bullets }
  } catch { return { text: '', bullets: [] } }
}

async function buildWithAI(input: { name: string; brandHint?: string; pages: Array<{ url: string; text: string; bullets: string[] }> }) {
  const system = `Eres un asistente experto en catálogo para PrestaShop. Devuelves JSON con:
- description_short (<= 800 caracteres)
- description_html (HTML con <p> y <ul>)
- features (array de {name, value})
- meta_title (<=70) y meta_description (<=160)
No inventes datos; prioriza página oficial o coincidencias múltiples.`
  const user = `Producto: ${input.name}
Marca (si se conoce): ${input.brandHint || ''}
Fuentes:
${input.pages.map(p => `URL: ${p.url}
${p.text.slice(0, 1200)}`).join('\n---\n')}`

  const resp = await OPENAI.chat.completions.create({
    model: 'gpt-4o-mini',
    messages: [
      { role: 'system', content: system },
      { role: 'user', content: user }
    ],
    temperature: 0.3,
    response_format: { type: 'json_object' }
  })
  return JSON.parse(resp.choices[0]?.message?.content || '{}')
}

function featuresToPipe(feats: Array<{name: string; value: string}>) {
  return feats.map(f => `${f.name}:${f.value}`).join(' | ')
}
function featuresToUL(feats: Array<{name: string; value: string}>) {
  return `<ul>\n${feats.map(f => `<li><strong>${f.name}:</strong> ${f.value}</li>`).join('\n')}\n</ul>`
}
function featuresToTable(feats: Array<{name: string; value: string}>) {
  const rows = feats.map(f => `<tr><th style="text-align:left;padding:4px 8px;">${f.name}</th><td style="padding:4px 8px;">${f.value}</td></tr>`).join('\n')
  return `<table class="ps-product-specs" style="border-collapse:collapse;width:100%;"><tbody>${rows}</tbody></table>`
}

export async function POST(req: NextRequest) {
  const { name, reference, ean13, mpn, brandHint } = await req.json()
  if (!name) return NextResponse.json({ error: 'Falta el nombre del producto' }, { status: 400 })

  const q = `${name} ficha técnica especificaciones sitio oficial`
  const pages = await webSearch(q)
  const top = pages.slice(0, 5)
  const texts = await Promise.all(top.map(async (p) => ({ url: p.url, ...(await extractText(p.url)) })))

  const ai = await buildWithAI({ name, brandHint, pages: texts })

  const imgCandidates = await imageSearch(`${name} product white background`)
  const validated: string[] = []
  for (const it of imgCandidates) {
    if (it.url && (it.width >= MIN_W && it.height >= MIN_H)) {
      const ok = await validateWhiteBgAndSize(it.url)
      if (ok) validated.push(it.url)
    }
    if (validated.length >= 6) break
  }

  const payload: any = {
    reference: reference || '',
    name,
    link_rewrite: psSlug(name),
    meta_title: ai.meta_title || name.slice(0, 70),
    meta_description: ai.meta_description || `${name} — compra en línea`,

    description_short: ai.description_short,
    description: ai.description_html,

    ean13: ean13 || '', mpn: mpn || '',
    features: featuresToPipe(ai.features || []),
    features_html: featuresToUL(ai.features || []),
    features_table_html: featuresToTable(ai.features || []),

    cover_image_url: validated[0] || '',
    image_urls: validated.join(','),

    categories: '', default_category: '', tags: '',
    active: 1, visibility: 'both', price_tax_excl: '', id_tax_rules_group: '', wholesale_price: '',
    on_sale: 0, ecotax: 0, quantity: '', minimal_quantity: 1, available_for_order: 1,
    weight: '', width: '', height: '', depth: '', meta_keywords: '',
    upc: '', isbn: '', supplier_reference: '', brand: brandHint || '', additional_shipping_cost: 0, attributes: ''
  }
  return NextResponse.json(payload)
}
