
'use client'
import * as XLSX from 'xlsx'
import { useState } from 'react'
import { apiPath } from '../lib/paths'

const OUTPUT_COLUMNS = [
  "reference","name","link_rewrite","categories","default_category","tags","active","visibility",
  "price_tax_excl","id_tax_rules_group","wholesale_price","on_sale","ecotax",
  "quantity","minimal_quantity","available_for_order",
  "weight","width","height","depth",
  "meta_title","meta_description","meta_keywords",
  "description_short","description",
  "ean13","upc","isbn","mpn","supplier_reference","brand",
  "additional_shipping_cost",
  "cover_image_url","image_urls",
  "attributes","features","features_html","features_table_html"
];

export default function BatchPage() {
  const [rows, setRows] = useState<any[]>([])
  const [processing, setProcessing] = useState(false)
  const [progress, setProgress] = useState(0)
  const [error, setError] = useState('')

  async function handleFile(e: React.ChangeEvent<HTMLInputElement>) {
    setError('')
    const file = e.target.files?.[0]
    if (!file) return
    const data = await file.arrayBuffer()
    const wb = XLSX.read(data, { type: 'array' })
    const ws = wb.Sheets['INPUT'] || wb.Sheets[wb.SheetNames[0]]
    if (!ws) { setError('No se encontró hoja INPUT.'); return }
    const json = XLSX.utils.sheet_to_json(ws, { defval: '' })
    const refs = json.map((r: any)=> ({ name: r.name, reference: r.reference || '', ean13: r.ean13 || '', mpn: r.mpn || '', brandHint: r.brand || '' }))
    setRows(refs)
  }

  async function runBatch() {
    setProcessing(true); setProgress(0); setError('')
    const outputs: any[] = []
    let done = 0

    for (const r of rows) {
      try {
        const res = await fetch(apiPath('/api/enrich'), { method: 'POST', headers: {'Content-Type':'application/json'}, body: JSON.stringify(r) })
        const j = await res.json()
        if (!res.ok) throw new Error(j.error || 'Error')
        const row: any = {}
        OUTPUT_COLUMNS.forEach((c) => row[c] = j[c] ?? '')
        outputs.push(row)
      } catch (e:any) {
        const row: any = {}
        OUTPUT_COLUMNS.forEach((c) => row[c] = '')
        row.name = r.name
        outputs.push(row)
      } finally {
        done += 1; setProgress(Math.round(done * 100 / rows.length))
        await new Promise(res=>setTimeout(res, 400))
      }
    }

    const wb = XLSX.utils.book_new()
    const ws = XLSX.utils.json_to_sheet(outputs)
    XLSX.utils.book_append_sheet(wb, ws, 'OUTPUT_PRESTASHOP')
    XLSX.writeFile(wb, 'Prestashop_READY_Batch.xlsx')
    setProcessing(false)
  }

  return (
    <div style={{ maxWidth: 700, margin: '0 auto', padding: 24 }}>
      <h1>Modo Batch (Excel)</h1>
      <p>Sube tu archivo con hoja INPUT (columna name).</p>
      <div style={{ background: '#fff', padding: 16, borderRadius: 12 }}>
        <input type="file" accept=".xlsx,.xls" onChange={handleFile} />
        <div style={{ marginTop: 12, display: 'flex', gap: 8 }}>
          <button onClick={runBatch} disabled={processing || rows.length===0}>{processing ? `Procesando… ${progress}%` : 'Procesar y Exportar'}</button>
          <a href="/productos">Volver</a>
        </div>
        {error && <div style={{ color: 'red', marginTop: 8 }}>{error}</div>}
        {rows.length>0 && <div style={{ color: '#555', marginTop: 8 }}>Productos cargados: {rows.length}</div>}
      </div>
    </div>
  )
}
