
'use client'
import { useState } from 'react'
import * as XLSX from 'xlsx'
import { apiPath } from './lib/paths'

const OUTPUT_COLUMNS = [
  "reference","name","link_rewrite","categories","default_category","tags","active","visibility",
  "price_tax_excl","id_tax_rules_group","wholesale_price","on_sale","ecotax",
  "quantity","minimal_quantity","available_for_order",
  "weight","width","height","depth",
  "meta_title","meta_description","meta_keywords",
  "description_short","description",
  "ean13","upc","isbn","mpn","supplier_reference","brand",
  "additional_shipping_cost",
  "cover_image_url","image_urls",
  "attributes","features","features_html","features_table_html"
];

export default function Page() {
  const [name, setName] = useState('')
  const [reference, setReference] = useState('')
  const [ean13, setEan] = useState('')
  const [brandHint, setBrandHint] = useState('')
  const [loading, setLoading] = useState(false)
  const [data, setData] = useState<any | null>(null)
  const [error, setError] = useState('')

  async function handleSubmit(e: React.FormEvent) {
    e.preventDefault()
    setLoading(true); setError(''); setData(null)
    try {
      const r = await fetch(apiPath('/api/enrich'), {
        method: 'POST', headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ name, reference, ean13, brandHint })
      })
      const j = await r.json()
      if (!r.ok) throw new Error(j.error || 'Error')
      setData(j)
    } catch (e: any) {
      setError(e.message)
    } finally { setLoading(false) }
  }

  function copy(text: string) { navigator.clipboard.writeText(text) }

  function exportXLSX() {
    if (!data) return
    const row: any = {}
    OUTPUT_COLUMNS.forEach((c) => row[c] = data[c] ?? '')
    const wb = XLSX.utils.book_new()
    const ws = XLSX.utils.json_to_sheet([row])
    XLSX.utils.book_append_sheet(wb, ws, 'OUTPUT_PRESTASHOP')
    XLSX.writeFile(wb, 'Prestashop_READY_Single.xlsx')
  }

  return (
    <div style={{ maxWidth: 900, margin: '0 auto', padding: 24 }}>
      <h1>PrestaShop Product Enricher</h1>
      <p>Ingresa un producto y obtén descripciones, ficha técnica e imágenes listas para PrestaShop.</p>

      <form onSubmit={handleSubmit} style={{ display: 'grid', gap: 8, marginTop: 16 }}>
        <input placeholder="Nombre del producto *" value={name} onChange={e=>setName(e.target.value)} required />
        <input placeholder="SKU / Reference (opcional)" value={reference} onChange={e=>setReference(e.target.value)} />
        <input placeholder="EAN13 (opcional)" value={ean13} onChange={e=>setEan(e.target.value)} />
        <input placeholder="Marca sugerida (opcional)" value={brandHint} onChange={e=>setBrandHint(e.target.value)} />
        <div>
          <button disabled={loading} type="submit">{loading ? 'Buscando…' : 'Buscar y rellenar'}</button>
          {data && (<>
            <button type="button" onClick={exportXLSX} style={{ marginLeft: 8 }}>Exportar XLSX</button>
            <button type="button" onClick={()=>copy(JSON.stringify(data))} style={{ marginLeft: 8 }}>Copiar JSON</button>
          </>)}
        </div>
      </form>

      {error && <div style={{ color: 'red', marginTop: 8 }}>{error}</div>}

      {data && (
        <div style={{ marginTop: 16, display: 'grid', gap: 16 }}>
          <section style={{ background: '#fff', padding: 16 }}>
            <h2>SEO & Slug</h2>
            <div>link_rewrite: <input value={data.link_rewrite} readOnly /> <button onClick={()=>copy(data.link_rewrite)}>Copiar</button></div>
            <div>meta_title: <input value={data.meta_title} readOnly /> <button onClick={()=>copy(data.meta_title)}>Copiar</button></div>
            <div>meta_description:</div>
            <textarea rows={2} readOnly value={data.meta_description} />
          </section>

          <section style={{ background: '#fff', padding: 16 }}>
            <h2>Descripciones</h2>
            <div>description_short</div>
            <textarea rows={2} readOnly value={data.description_short} />
            <div>description (HTML)</div>
            <textarea rows={6} readOnly value={data.description} />
          </section>

          <section style={{ background: '#fff', padding: 16 }}>
            <h2>Características</h2>
            <div>features (pipe)</div>
            <textarea rows={2} readOnly value={data.features} />
            <div style={{ display: 'grid', gap: 8, gridTemplateColumns: '1fr 1fr' }}>
              <div>
                <div>features_html</div>
                <textarea rows={6} readOnly value={data.features_html} />
              </div>
              <div>
                <div>features_table_html</div>
                <textarea rows={6} readOnly value={data.features_table_html} />
              </div>
            </div>
          </section>

          <section style={{ background: '#fff', padding: 16 }}>
            <h2>Imágenes</h2>
            <div>cover_image_url</div>
            <input readOnly value={data.cover_image_url} />
            <div>image_urls</div>
            <textarea rows={2} readOnly value={data.image_urls} />
          </section>

          <section style={{ background: '#fff', padding: 16 }}>
            <a href="/productos/batch">Ir a modo Batch (Excel)</a>
          </section>
        </div>
      )}
    </div>
  )
}
